#include <iostream>
#include <iomanip>
#include <complex>
#include <cmath>
#include <limits>

using namespace std;

class Complex {
  private:
    double r, phi;

  public:
    Complex(double Re=0, double Im=0):
      r(sqrt(Re*Re + Im*Im)), phi(atan2(Im, Re)) {}

    friend Complex operator+(Complex z1, Complex z2) {
      Complex z;
      z.r = sqrt(z1.r*z1.r + z2.r*z2.r + 2*z1.r*z2.r*cos(z1.phi - z2.phi));
      z.phi = atan2(z1.r*sin(z1.phi) + z2.r*sin(z2.phi),
                    z1.r*cos(z1.phi) + z2.r*cos(z2.phi));
      return z;
    }
    
    friend Complex operator-(Complex z1, Complex z2) {
      Complex z;
      z.r = sqrt(z1.r*z1.r + z2.r*z2.r - 2*z1.r*z2.r*cos(z1.phi - z2.phi));
      z.phi = atan2(z1.r*sin(z1.phi) - z2.r*sin(z2.phi),
                    z1.r*cos(z1.phi) - z2.r*cos(z2.phi));
      return z;
    }
    
    friend Complex operator*(Complex z1, Complex z2) {
      Complex z;
      z.r = z1.r * z2.r;
      z.phi = z1.phi + z2.phi;
    
      if (z.phi >  M_PI) z.phi -= 2*M_PI;
      if (z.phi < -M_PI) z.phi += 2*M_PI;
    
      return z;
    }
    
    friend Complex operator/(Complex z1, Complex z2) {
      Complex z;
      z.r = z1.r / z2.r;
      z.phi = z1.phi - z2.phi;
    
      if (z.phi >  M_PI) z.phi -= 2*M_PI;
      if (z.phi < -M_PI) z.phi += 2*M_PI;
    
      return z;
    }

    friend Complex sqrt(Complex z1) {
      Complex z;
      z.r = sqrt(z1.r);
      z.phi = z1.phi / 2;
      return z;
    }
    
    friend Complex exp(Complex z1) {
      Complex z;
      z.r = exp(z1.r * cos(z1.phi));
      z.phi = z1.r * sin(z1.phi);
    
      z.phi = fmod(z.phi, 2*M_PI);
      if (z.phi >  M_PI) z.phi -= 2*M_PI;
      if (z.phi < -M_PI) z.phi += 2*M_PI;
    
      return z;
    }
    
    friend Complex log(Complex z1) {
      return Complex{log(z1.r), z1.phi};
    }
    
    friend Complex pow(Complex z1, Complex z2) {
      return exp(z2 * log(z1));
    }

    double real() const {
      return r * cos(phi);
    }
    
    double imag() const {
      return r * sin(phi);
    }
    
    friend ostream& operator<<(ostream& stream, Complex z) {
      return stream << "(" << z.real() << "," << z.imag() << ")";
    }
};

int main()
{
  complex<double> w1,w2;

  cout << "z1 z2: ";
  cin >> w1 >> w2;

  Complex z1{w1.real(), w1.imag()}, z2{w2.real(), w2.imag()};

  cout << fixed << setprecision(numeric_limits<double>::digits10);
  cout << "(z1-z2)/(z1+z2):" << endl
       << "  " << (z1-z2)/(z1+z2) << " " << (w1-w2)/(w1+w2) << endl;
  cout << "sqrt(z1*z2):" << endl
       << "  " << sqrt(z1*z2) << " " << sqrt(w1*w2) << endl;
  cout << "exp(z1):" << endl
       << "  " << exp(z1) << " " << exp(w1) << endl;
  cout << "log(z2):" << endl
       << "  " << log(z2) << " " << log(w2) << endl;
  cout << "pow(z1,z2):" << endl
       << "  " << pow(z1,z2) << " " << pow(w1,w2) << endl;

  return 0;
}
